<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Models\Brand;
use App\Models\Category;
use App\Models\Subcategory;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

class BrandController extends Controller
{
    public function index()
    {
        $brands = Brand::orderBy('id', 'desc')->paginate(10);
        return Inertia::render('Admin/Brands', [
            'brands' => $brands,
        ]);
    }

    public function create()
    {
        return Inertia::render('Admin/Brands/Create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => 'required|string|max:255',
            'logo' => 'nullable|image|max:2048',
        ]);

        $brandData = [
            'name' => $data['name'],
            'slug' => Str::slug($data['name']),
        ];

        // if ($request->hasFile('logo')) {
        //     $path = $request->file('logo')->store('brands', 'public');
        //     $brandData['logo'] = $path;
        // }

        if ($request->hasFile('logo')) {
            $logo = $request->file('logo');
            $filename = time() . '_' . $logo->getClientOriginalName();
            $path = $logo->storeAs('brands', $filename, 'public');
            $brandData['logo'] = $path;
        }

        $brand = Brand::create($brandData);

        return redirect()->route('admin.brands.index')->with('success', 'Brand created.');
    }

    public function show($id)
    {
        return Inertia::render('Admin/Brands/Show', ['id' => $id]);
    }

    public function edit($id)
    {
        $brand = Brand::findOrFail($id);
        return Inertia::render('Admin/Brands/Edit', ['brand' => $brand]);
    }

    public function update(Request $request, $id)
    {
        $brand = Brand::findOrFail($id);

        $data = $request->validate([
            'name' => 'sometimes|required|string|max:255',
            'logo' => 'nullable|image|max:2048',
        ]);

        $brandData = [];

        if ($request->filled('name')) {
            $brandData['name'] = $data['name'];
            $brandData['slug'] = Str::slug($data['name']);
        }

        // if ($request->hasFile('logo')) {
        //     $path = $request->file('logo')->store('brands', 'public');
        //     $brandData['logo'] = $path;
        // }

        if ($request->hasFile('logo')) {
            // Delete old logo if exists
            if ($brand->logo && \Storage::disk('public')->exists($brand->logo)) {
                \Storage::disk('public')->delete($brand->logo);
            }

            $logo = $request->file('logo');
            $filename = time() . '_' . $logo->getClientOriginalName();
            $path = $logo->storeAs('brands', $filename, 'public');
            $brandData['logo'] = $path;
        }

        if (!empty($brandData)) {
            $brand->update($brandData);
        }

        return redirect()->route('admin.brands.index')->with('success', 'Brand updated.');
    }

    public function destroy($id)
    {
        $brand = Brand::findOrFail($id);

        $categories = Category::where('brand_id', $brand->id)->pluck('name')->toArray();
        $subcategories = Subcategory::where('brand_id', $brand->id)->pluck('name')->toArray();

        if (!empty($categories) || !empty($subcategories)) {
            $parts = [];
            if (!empty($categories)) {
                $parts[] = 'categories: ' . implode(', ', $categories);
            }
            if (!empty($subcategories)) {
                $parts[] = 'subcategories: ' . implode(', ', $subcategories);
            }

            // Delete logo if exists
            if ($brand->logo && \Storage::disk('public')->exists($brand->logo)) {
                \Storage::disk('public')->delete($brand->logo);
            }

            return redirect()->route('admin.brands.index')->with('error', 'Cannot delete brand; related ' . implode('; ', $parts) . '.');
        }

        $brand->delete();
        return redirect()->route('admin.brands.index')->with('success', 'Brand deleted.');
    }

    public function blockers($id)
    {
        $brand = Brand::findOrFail($id);
        $categories = Category::where('brand_id', $brand->id)->pluck('name')->toArray();
        $subcategories = Subcategory::where('brand_id', $brand->id)->pluck('name')->toArray();

        $blocked = !empty($categories) || !empty($subcategories);

        return response()->json([
            'blocked' => $blocked,
            'categories' => $categories,
            'subcategories' => $subcategories,
        ]);
    }
}
