<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Models\Order;
use App\Models\Product;
use App\Models\OrderItem;

class OrderController extends Controller
{
    public function index()
    {
        $orders = Order::with('items.product')->where('soft_delete', 0)->orderBy('id', 'desc')->paginate(10);
        return Inertia::render('Admin/Orders', ['orders' => $orders]);
    }

    public function create()
    {
        $products = Product::where('status', 'active')->orderBy('name')->get(['id', 'name', 'price', 'stock_quantity']);
        return Inertia::render('Admin/Orders/Create', ['products' => $products]);
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'customer_name' => 'required|string|max:255',
            'customer_email' => 'nullable|email|max:255',
            'customer_phone' => 'nullable|string|max:20',
            'customer_address' => 'nullable|string',
            'shipping_cost' => 'nullable|numeric|min:0',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.price' => 'required|numeric|min:0',
        ]);

        $order = Order::create([
            'order_number' => Order::generateOrderNumber(),
            'customer_name' => $data['customer_name'],
            'customer_email' => $data['customer_email'],
            'customer_phone' => $data['customer_phone'],
            'customer_address' => $data['customer_address'],
            'subtotal' => 0,
            'tax_amount' => 0,
            'shipping_cost' => $data['shipping_cost'] ?? 0,
            'total_amount' => 0,
        ]);

        $subtotal = 0;
        foreach ($data['items'] as $item) {
            $total = $item['quantity'] * $item['price'];
            $subtotal += $total;
            
            OrderItem::create([
                'order_id' => $order->id,
                'product_id' => $item['product_id'],
                'quantity' => $item['quantity'],
                'price' => $item['price'],
                'total' => $total,
            ]);
        }

        $order->update([
            'subtotal' => $subtotal,
            'total_amount' => $subtotal + ($order->shipping_cost ?? 0),
        ]);

        // Auto-generate cash in voucher
        \App\Models\Voucher::create([
            'voucher_number' => \App\Models\Voucher::generateVoucherNumber('cash_in'),
            'type' => 'cash_in',
            'source' => 'auto',
            'amount' => $subtotal + ($order->shipping_cost ?? 0),
            'description' => 'Payment received for order ' . $order->order_number,
            'order_id' => $order->id,
        ]);

        return redirect()->route('admin.orders.index')->with('success', 'Order created successfully.');
    }

    public function show($id)
    {
        $order = Order::with(['items.product', 'user'])->findOrFail($id);
        return Inertia::render('Admin/Orders/Show', ['order' => $order]);
    }

    public function edit($id)
    {
        $order = Order::with('items.product')->findOrFail($id);
        $products = Product::where('status', 'active')->orderBy('name')->get(['id', 'name', 'price', 'stock_quantity']);
        return Inertia::render('Admin/Orders/Edit', ['order' => $order, 'products' => $products]);
    }

    public function update(Request $request, $id)
    {
        $order = Order::findOrFail($id);
        
        $data = $request->validate([
            'customer_name' => 'required|string|max:255',
            'customer_email' => 'nullable|email|max:255',
            'customer_phone' => 'nullable|string|max:20',
            'customer_address' => 'nullable|string',
            'status' => 'required|in:pending,confirmed,shipped,delivered,cancelled',
            'shipping_cost' => 'nullable|numeric|min:0',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.price' => 'required|numeric|min:0',
        ]);

        $order->update([
            'customer_name' => $data['customer_name'],
            'customer_email' => $data['customer_email'],
            'customer_phone' => $data['customer_phone'],
            'customer_address' => $data['customer_address'],
            'status' => $data['status'],
            'shipping_cost' => $data['shipping_cost'] ?? 0,
        ]);

        // Handle voucher updates based on status changes
        $this->handleVoucherUpdates($order, $data['status']);

        $order->items()->delete();
        
        $subtotal = 0;
        foreach ($data['items'] as $item) {
            $total = $item['quantity'] * $item['price'];
            $subtotal += $total;
            
            OrderItem::create([
                'order_id' => $order->id,
                'product_id' => $item['product_id'],
                'quantity' => $item['quantity'],
                'price' => $item['price'],
                'total' => $total,
            ]);
        }

        $order->update([
            'subtotal' => $subtotal,
            'total_amount' => $subtotal + $order->shipping_cost,
        ]);

        return redirect()->route('admin.orders.index')->with('success', 'Order updated successfully.');
    }

    public function destroy($id)
    {
        $order = Order::findOrFail($id);
        
        // Soft delete the order
        $order->update(['soft_delete' => 1]);
        
        // Delete all auto vouchers related to this order
        \App\Models\Voucher::where('order_id', $order->id)
            ->where('source', 'auto')
            ->delete();
            
        return redirect()->route('admin.orders.index')->with('success', 'Order deleted successfully.');
    }

    public function invoice($id)
    {
        $order = Order::with(['items.product', 'user'])->findOrFail($id);
        return Inertia::render('Admin/Orders/Invoice', ['order' => $order]);
    }

    private function handleVoucherUpdates($order, $newStatus)
    {
        $oldStatus = $order->getOriginal('status');
        
        // If order is cancelled, create refund voucher and remove profit voucher
        if ($newStatus === 'cancelled') {
            // Delete all existing auto vouchers for this order
            \App\Models\Voucher::where('order_id', $order->id)
                ->where('source', 'auto')
                ->delete();
                
            // Create refund voucher (cash out)
            \App\Models\Voucher::create([
                'voucher_number' => \App\Models\Voucher::generateVoucherNumber('cash_out'),
                'type' => 'cash_out',
                'source' => 'auto',
                'amount' => $order->total_amount,
                'description' => 'Refund for cancelled order ' . $order->order_number,
                'order_id' => $order->id,
            ]);
        }
        
        // If order status changed from cancelled to active, remove refund voucher
        if ($oldStatus === 'cancelled' && $newStatus !== 'cancelled') {
            \App\Models\Voucher::where('order_id', $order->id)
                ->where('source', 'auto')
                ->delete();
                
            // Recreate payment voucher
            \App\Models\Voucher::create([
                'voucher_number' => \App\Models\Voucher::generateVoucherNumber('cash_in'),
                'type' => 'cash_in',
                'source' => 'auto',
                'amount' => $order->total_amount,
                'description' => 'Payment received for order ' . $order->order_number,
                'order_id' => $order->id,
            ]);
        }
        
        // Generate profit voucher when order is delivered
        if ($newStatus === 'delivered' && $oldStatus !== 'delivered') {
            $totalCost = 0;
            foreach ($order->items as $item) {
                $totalCost += $item->quantity * ($item->product->cost_price ?? 0);
            }
            
            $profit = $order->total_amount - $totalCost;
            
            if ($profit > 0) {
                \App\Models\Voucher::create([
                    'voucher_number' => \App\Models\Voucher::generateVoucherNumber('cash_in'),
                    'type' => 'cash_in',
                    'source' => 'auto',
                    'amount' => $profit,
                    'description' => 'Profit from completed order ' . $order->order_number,
                    'order_id' => $order->id,
                ]);
            }
        }
        
        // Remove profit voucher if order status changed from delivered to non-delivered
        if ($oldStatus === 'delivered' && $newStatus !== 'delivered') {
            \App\Models\Voucher::where('order_id', $order->id)
                ->where('description', 'like', 'Profit from completed order%')
                ->delete();
        }
    }
}
