<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Inertia\Inertia;
use App\Models\StockHistory;
use App\Models\Product;
use App\Models\Order;
use Illuminate\Http\Request;

class ReportController extends Controller
{
    public function index()
    {
        return Inertia::render('Admin/Reports');
    }

    public function stockHistory(Request $request)
    {
        $query = StockHistory::with('product');
        
        if ($request->product_id) {
            $query->where('product_id', $request->product_id);
        }
        
        if ($request->date_from) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }
        
        if ($request->date_to) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }
        
        $histories = $query->orderBy('created_at', 'desc')->paginate(20)->withQueryString();
        
        return Inertia::render('Admin/Reports/StockHistory', [
            'histories' => $histories,
            'filters' => $request->only(['product_id', 'date_from', 'date_to']),
            'products' => Product::orderBy('name')->get(['id', 'name'])
        ]);
    }

    public function ordersProfit(Request $request)
    {
        $query = Order::with(['items.product'])->where('status', '!=', 'cancelled')->where('soft_delete', 0);
        
        if ($request->date_from) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }
        
        if ($request->date_to) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }
        
        if ($request->status) {
            $query->where('status', $request->status);
        }
        
        $orders = $query->orderBy('created_at', 'desc')->paginate(20)->withQueryString();
        
        // Calculate profit for each order
        $orders->getCollection()->transform(function ($order) {
            $totalCost = 0;
            $totalRevenue = $order->total_amount;
            
            foreach ($order->items as $item) {
                $totalCost += $item->quantity * ($item->product->cost_price ?? 0);
            }
            
            $order->total_cost = $totalCost;
            $order->profit = $totalRevenue - $totalCost;
            $order->profit_margin = $totalRevenue > 0 ? ($order->profit / $totalRevenue) * 100 : 0;
            
            return $order;
        });
        
        return Inertia::render('Admin/Reports/OrdersProfit', [
            'orders' => $orders,
            'filters' => $request->only(['date_from', 'date_to', 'status'])
        ]);
    }
}
