<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Models\Subcategory;
use App\Models\Category;
use App\Models\Brand;
use App\Models\Product;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Str;

class SubcategoryController extends Controller
{
    public function index()
    {
        $subcategories = Subcategory::with(['category','brand'])->orderBy('id','desc')->paginate(10);
        return Inertia::render('Admin/Subcategories/Index', ['subcategories' => $subcategories]);
    }

    /**
     * Generate a unique slug for the given name and optional brand.
     * If $ignoreId is provided, exclude that record (useful for updates).
     * Examples: "vegetable" or "vegetable-china".
     */
    private function makeUniqueSlug(string $name, $brandId = null, $ignoreId = null)
    {
        $base = Str::slug($name);

        if ($brandId) {
            $brand = Brand::find($brandId);
            $brandPart = $brand ? Str::slug($brand->name) : null;
        } else {
            $brandPart = null;
        }

        $candidateBase = $brandPart ? ($base . '-' . $brandPart) : $base;
        $slug = $candidateBase;
        $i = 1;

        while (Subcategory::where('slug', $slug)->when($ignoreId, function ($q) use ($ignoreId) {
            return $q->where('id', '!=', $ignoreId);
        })->exists()) {
            $slug = $candidateBase . '-' . $i++;
        }

        return $slug;
    }

    public function create()
    {
        $categories = Category::orderBy('name')->get();
        $brands = Brand::orderBy('name')->get();
        return Inertia::render('Admin/Subcategories/Create', [
            'categories' => $categories,
            'brands' => $brands,
        ]);
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => 'required|string|max:255',
            'category_id' => 'nullable|exists:categories,id',
            'brand_id' => 'nullable|exists:brands,id',
        ]);

        Subcategory::create([
            'name' => $data['name'],
            'slug' => $this->makeUniqueSlug($data['name'], $data['brand_id'] ?? null),
            'category_id' => $data['category_id'] ?? null,
            'brand_id' => $data['brand_id'] ?? null,
        ]);

        return redirect()->route('admin.subcategories.index')->with('success','Subcategory created.');
    }

    public function edit($id)
    {
        $subcategory = Subcategory::findOrFail($id);
        $categories = Category::orderBy('name')->get();
        $brands = Brand::orderBy('name')->get();
        return Inertia::render('Admin/Subcategories/Edit', [
            'subcategory' => $subcategory,
            'categories' => $categories,
            'brands' => $brands,
        ]);
    }

    public function update(Request $request, $id)
    {
        $subcategory = Subcategory::findOrFail($id);

        $data = $request->validate([
            'name' => 'required|string|max:255',
            'category_id' => 'nullable|exists:categories,id',
            'brand_id' => 'nullable|exists:brands,id',
        ]);

        $subcategory->update([
            'name' => $data['name'],
            'slug' => $this->makeUniqueSlug($data['name'], $data['brand_id'] ?? null, $subcategory->id),
            'category_id' => $data['category_id'] ?? null,
            'brand_id' => $data['brand_id'] ?? null,
        ]);

        return redirect()->route('admin.subcategories.index')->with('success','Subcategory updated.');
    }

    public function destroy($id)
    {
        $sub = Subcategory::findOrFail($id);

        // If products table has subcategory_id, check for products referencing this subcategory
        if (Schema::hasColumn('products', 'subcategory_id')) {
            $products = Product::where('subcategory_id', $sub->id)->pluck('name')->toArray();
            if (!empty($products)) {
                return redirect()->route('admin.subcategories.index')->with('error', 'Cannot delete subcategory; related products: ' . implode(', ', $products) . '.');
            }
        }

        $sub->delete();
        return redirect()->route('admin.subcategories.index')->with('success','Subcategory deleted.');
    }

    public function blockers($id)
    {
        $sub = Subcategory::findOrFail($id);

        $products = [];
        if (\Illuminate\Support\Facades\Schema::hasColumn('products', 'subcategory_id')) {
            $products = Product::where('subcategory_id', $sub->id)->pluck('name')->toArray();
        }

        $blocked = !empty($products);

        return response()->json([
            'blocked' => $blocked,
            'products' => $products,
        ]);
    }
}
