<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Models\Voucher;

class VoucherController extends Controller
{
    public function index(Request $request)
    {
        $query = Voucher::with('order');
        
        if ($request->type) {
            $query->where('type', $request->type);
        }
        
        if ($request->source) {
            $query->where('source', $request->source);
        }
        
        if ($request->date_from) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }
        
        if ($request->date_to) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }
        
        $vouchers = $query->orderBy('created_at', 'desc')->paginate(20)->withQueryString();
        
        // Calculate cash balance
        $cashIn = Voucher::where('type', 'cash_in')->sum('amount');
        $cashOut = Voucher::where('type', 'cash_out')->sum('amount');
        $balance = $cashIn - $cashOut;
        
        return Inertia::render('Admin/Vouchers', [
            'vouchers' => $vouchers,
            'filters' => $request->only(['type', 'source', 'date_from', 'date_to']),
            'balance' => $balance,
            'cashIn' => $cashIn,
            'cashOut' => $cashOut
        ]);
    }

    public function create()
    {
        return Inertia::render('Admin/Vouchers/Create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'type' => 'required|in:cash_in,cash_out',
            'amount' => 'required|numeric|min:0.01',
            'description' => 'required|string|max:255',
            'reference' => 'nullable|string|max:255',
        ]);

        Voucher::create([
            'voucher_number' => Voucher::generateVoucherNumber($data['type']),
            'type' => $data['type'],
            'source' => 'manual',
            'amount' => $data['amount'],
            'description' => $data['description'],
            'reference' => $data['reference'],
        ]);

        return redirect()->route('admin.vouchers.index')->with('success', 'Voucher created successfully.');
    }

    public function show($id)
    {
        $voucher = Voucher::with('order')->findOrFail($id);
        return Inertia::render('Admin/Vouchers/Show', ['voucher' => $voucher]);
    }

    public function destroy($id)
    {
        $voucher = Voucher::findOrFail($id);
        
        if ($voucher->source === 'auto') {
            return back()->withErrors(['error' => 'Cannot delete auto-generated vouchers.']);
        }
        
        $voucher->delete();
        return redirect()->route('admin.vouchers.index')->with('success', 'Voucher deleted successfully.');
    }
}
