<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\Category;
use App\Models\Subcategory;
use App\Models\Brand;
use App\Models\Unit;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use App\Models\Deal;
use App\Models\Notification;

class ProductIndexController extends Controller
{
    public function index()
    {
        $categories = Category::with(['subcategories', 'brand'])
            ->orderBy('name')
            ->get();
            
        $recentProducts = Product::where('status', 'active')
            ->where('visibility', 'public')
            ->with(['category', 'subcategory', 'brand'])
            ->orderBy('created_at', 'desc')
            ->limit(6)
            ->get();
            
        $products = Product::where('status', 'active')
            ->where('visibility', 'public')
            ->with(['category', 'subcategory', 'brand'])
            ->orderBy('name')
            ->paginate(12);
        $activeDeal = Deal::with('product')->active()->first();
        
        $notifications = Auth::check() ? 
            Notification::where('user_id', Auth::id())
                ->orWhereNull('user_id')
                ->orderBy('created_at', 'desc')
                ->limit(5)
                ->get() : collect();
            
        return view('index', compact('products', 'categories', 'recentProducts', 'activeDeal', 'notifications'));
    }

    public function show($id)
    {
        $categories = Category::with(['subcategories', 'brand'])->orderBy('name')->get();
        $product = Product::with(['category', 'subcategory', 'brand', 'unit'])
            ->findOrFail($id);
        
        $notifications = Auth::check() ? 
            Notification::where('user_id', Auth::id())
                ->orWhereNull('user_id')
                ->orderBy('created_at', 'desc')
                ->limit(5)
                ->get() : collect();
            
        return view('productdetails', compact('product', 'categories', 'notifications'));
    }

    public function search(Request $request)
    {
        $query = $request->get('q');
        
        $products = Product::where('status', 'active')
            ->where('visibility', 'public')
            ->where(function($q) use ($query) {
                $q->where('name', 'LIKE', "%{$query}%")
                  ->orWhere('description', 'LIKE', "%{$query}%")
                  ->orWhere('sku', 'LIKE', "%{$query}%");
            })
            ->with(['category', 'subcategory', 'brand'])
            ->paginate(12);
            
        $categories = Category::with(['subcategories', 'brand'])
            ->orderBy('name')
            ->get();
        
        $notifications = Auth::check() ? 
            Notification::where('user_id', Auth::id())
                ->orWhereNull('user_id')
                ->orderBy('created_at', 'desc')
                ->limit(5)
                ->get() : collect();
            
        return view('search', compact('products', 'categories', 'query', 'notifications'));
    }

    public function suggestions(Request $request)
    {
        $query = $request->get('q');
        
        if (strlen($query) < 2) {
            return response()->json([]);
        }
        
        $words = explode(' ', trim($query));
        
        $products = Product::where('status', 'active')
            ->where('visibility', 'public')
            ->where(function($q) use ($words) {
                foreach ($words as $word) {
                    $word = trim($word);
                    if (strlen($word) > 1) {
                        $q->where(function($subQuery) use ($word) {
                            $subQuery->where('name', 'LIKE', "%{$word}%")
                                     ->orWhere('description', 'LIKE', "%{$word}%")
                                     ->orWhereHas('brand', function($brandQuery) use ($word) {
                                         $brandQuery->where('name', 'LIKE', "%{$word}%");
                                     })
                                     ->orWhereHas('category', function($catQuery) use ($word) {
                                         $catQuery->where('name', 'LIKE', "%{$word}%");
                                     });
                        });
                    }
                }
            })
            ->with(['brand', 'category'])
            ->select('id', 'name', 'brand_id', 'category_id', 'price', 'discount_price')
            ->limit(8)
            ->get()
            ->map(function($product) {
                return [
                    'id' => $product->id,
                    'name' => $product->name,
                    'brand' => $product->brand->name ?? '',
                    'category' => $product->category->name ?? '',
                    'price' => $product->discount_price ?: $product->price
                ];
            });
            
        return response()->json($products);
    }

    public function allStore()
    {
        $categories = Category::with(['subcategories', 'brand'])->orderBy('name')->get();
        $brands = Brand::withCount('categories')->get();
        
        $notifications = Auth::check() ? 
            Notification::where('user_id', Auth::id())
                ->orWhereNull('user_id')
                ->orderBy('created_at', 'desc')
                ->limit(5)
                ->get() : collect();
                
        return view('allstore', compact('brands', 'categories', 'notifications'));
    }
    public function shopall(Request $request)
    {
        $categories = Category::with(['subcategories', 'brand'])->orderBy('name')->get();
        $brands = Brand::orderBy('name')->get();
        
        $query = Product::where('status', 'active')
            ->where('visibility', 'public')
            ->with(['category', 'subcategory', 'brand']);
            
        if ($request->category) {
            $query->whereHas('category', function($q) use ($request) {
                $q->where('id', $request->category);
            });
        }
        
        if ($request->brand) {
            $query->whereHas('brand', function($q) use ($request) {
                $q->where('id', $request->brand);
            });
        }
        
        if ($request->min_price) {
            $query->where('price', '>=', $request->min_price);
        }
        
        if ($request->max_price) {
            $query->where('price', '<=', $request->max_price);
        }
        
        if ($request->sort) {
            switch ($request->sort) {
                case 'price_low':
                    $query->orderBy('price', 'asc');
                    break;
                case 'price_high':
                    $query->orderBy('price', 'desc');
                    break;
                case 'name':
                    $query->orderBy('name', 'asc');
                    break;
                default:
                    $query->orderBy('created_at', 'desc');
            }
        } else {
            $query->orderBy('name');
        }
        
        $perPage = $request->per_page ?: 12;
        $products = $query->paginate($perPage)->withQueryString();
        
        $notifications = Auth::check() ? 
            Notification::where('user_id', Auth::id())
                ->orWhereNull('user_id')
                ->orderBy('created_at', 'desc')
                ->limit(5)
                ->get() : collect();
        
        return view('shopproducts', compact('products', 'categories', 'brands', 'notifications'));
    }
    public function filter($category, $subcategory = null)
    {
        $categories = Category::with(['subcategories', 'brand'])
            ->orderBy('name')
            ->get();

        $query = Product::where('status', 'active')
            ->where('visibility', 'public')
            ->with(['category', 'subcategory', 'brand'])
            ->orderBy('name');

        $query->whereHas('subcategory.category', function($q) use ($category) {
            $q->whereSlug($category);
        });

        if ($subcategory) {
            $query->whereHas('subcategory', function($q) use ($subcategory) {
                $q->whereSlug($subcategory);
            });
        }

        $products = $query->paginate(12);

        return view('shopproducts', compact('products', 'categories'));
    }
    
    public function addToCart(Request $request)
    {
        $product = Product::findOrFail($request->product_id);
        $cart = session()->get('cart', []);
        
        // Create unique cart key with weight if provided
        $cartKey = $request->weight ? $product->id . '_' . $request->weight : $product->id;
        
        if (isset($cart[$cartKey])) {
            $cart[$cartKey]['quantity'] += $request->quantity;
        } else {
            $cart[$cartKey] = [
                'product_id' => $product->id,
                'name' => $product->name,
                'price' => $product->discount_price ?: $product->price,
                'image' => $product->image,
                'quantity' => $request->quantity,
                'weight' => $request->weight ?? null
            ];
        }
        
        session()->put('cart', $cart);
        return response()->json(['success' => true, 'cart_count' => array_sum(array_column($cart, 'quantity'))]);
    }
    
    public function removeFromCart(Request $request)
    {
        $cart = session()->get('cart', []);
        unset($cart[$request->product_id]);
        session()->put('cart', $cart);
        return response()->json(['success' => true, 'cart_count' => array_sum(array_column($cart, 'quantity'))]);
    }
    
    public function updateCart(Request $request)
    {
        $cart = session()->get('cart', []);
        if (isset($cart[$request->product_id])) {
            $cart[$request->product_id]['quantity'] = $request->quantity;
            session()->put('cart', $cart);
        }
        return response()->json(['success' => true, 'cart_count' => array_sum(array_column($cart, 'quantity'))]);
    }
    
    public function getCartItems()
    {
        $cart = session()->get('cart', []);
        return response()->json($cart);
    }
    
    public function shopCheckout()
    {
        $cart = session()->get('cart', []);
        if (empty($cart)) {
            return redirect()->route('shop-all');
        }
        $categories = Category::with(['subcategories', 'brand'])->orderBy('name')->get();
        $addresses = Auth::check() ? Auth::user()->shippingAddresses : collect();
        
        $notifications = Auth::check() ? 
            Notification::where('user_id', Auth::id())
                ->orWhereNull('user_id')
                ->orderBy('created_at', 'desc')
                ->limit(5)
                ->get() : collect();
                
        return view('shopcheckout', compact('cart', 'categories', 'addresses', 'notifications'));
    }
    
    public function myOrders()
    {
        $categories = Category::with(['subcategories', 'brand'])->orderBy('name')->get();
        $orders = Order::where('user_id', Auth::id())
            ->with(['orderItems.product'])
            ->orderBy('created_at', 'desc')
            ->paginate(10);
        
        $notifications = Auth::check() ? 
            Notification::where('user_id', Auth::id())
                ->orWhereNull('user_id')
                ->orderBy('created_at', 'desc')
                ->limit(5)
                ->get() : collect();
            
        return view('myorders', compact('orders', 'categories', 'notifications'));
    }
    
    public function placeOrder(Request $request)
    {
        $cart = session()->get('cart', []);
        if (empty($cart)) {
            return redirect()->route('shop-all');
        }
        
        // Handle guest registration
        if (!Auth::check() && $request->guest_registration) {
            $request->validate([
                'name' => 'required|string|max:255',
                'email' => 'required|string|email|max:255|unique:users',
                'password' => 'required|string|min:8|confirmed',
            ]);
            
            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'password' => Hash::make($request->password),
                'role_id' => 1,
            ]);
            
            Auth::login($user);
        }
        
        if (!Auth::check()) {
            return redirect()->route('login')->with('message', 'Please login to place an order.');
        }
        
        $subtotal = 0;
        foreach ($cart as $item) {
            $subtotal += $item['price'] * $item['quantity'];
        }
        
        $order = Order::create([
            'user_id' => Auth::id(),
            'order_number' => Order::generateOrderNumber(),
            'customer_name' => Auth::user()->name,
            'customer_email' => Auth::user()->email,
            'delivery_date' => $request->delivery_date ?? 'today',
            'delivery_time' => $request->delivery_time ?? '9am-12pm',
            'delivery_instructions' => $request->delivery_instructions,
            'payment_method' => $request->payment_method ?? 'cash_on_delivery',
            'subtotal' => $subtotal,
            'total_amount' => $subtotal,
            'status' => 'pending'
        ]);
        
        foreach ($cart as $item) {
            OrderItem::create([
                'order_id' => $order->id,
                'product_id' => $item['product_id'],
                'quantity' => $item['quantity'],
                'price' => $item['price'],
                'total' => $item['price'] * $item['quantity']
            ]);
        }
        
        // Create order notification
        Notification::create([
            'user_id' => Auth::id(),
            'type' => 'order',
            'title' => 'Order Confirmed',
            'message' => "Your order #{$order->order_number} has been confirmed and is being processed.",
            'data' => ['order_id' => $order->id]
        ]);
        
        session()->forget('cart');
        
        return redirect()->route('home')->with('success', 'Order placed successfully! Order #' . $order->order_number);
    }
    
    public function markNotificationRead($id)
    {
        $notification = Notification::findOrFail($id);
        if ($notification->user_id === Auth::id() || $notification->user_id === null) {
            $notification->markAsRead();
        }
        return response()->json(['success' => true]);
    }
}
