<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;

class Product extends Model
{
    use HasFactory;

    protected $fillable = [
        'name', 'slug', 'price', 'category_id', 'subcategory_id', 'brand_id', 'unit_id', 'sku', 'cost_price', 'image',
        'stock_quantity', 'low_stock_threshold', 'stock_status', 'status', 'is_featured', 'visibility',
        'description', 'short_description', 'weight', 'dimensions', 'barcode',
        'discount_price', 'tax_rate', 'meta_title', 'meta_description', 'tags'
    ];

    protected $casts = [
        'is_featured' => 'boolean',
        'price' => 'decimal:2',
        'cost_price' => 'decimal:2',
        'discount_price' => 'decimal:2',
        'weight' => 'decimal:2',
        'tax_rate' => 'decimal:2',
    ];

    public function category()
    {
        return $this->belongsTo(Category::class);
    }

    public function subcategory()
    {
        return $this->belongsTo(Subcategory::class);
    }

    public function brand()
    {
        return $this->belongsTo(Brand::class);
    }

    public function unit()
    {
        return $this->belongsTo(Unit::class);
    }

    public function getImageUrlAttribute()
    {
        return $this->image ? Storage::url($this->image) : null;
    }

    public function getProfitMarginAttribute()
    {
        return $this->price - $this->cost_price;
    }

    public function getStockValueAttribute()
    {
        return $this->stock_quantity * $this->cost_price;
    }

    public function getFinalPriceAttribute()
    {
        return $this->discount_price ?? $this->price;
    }

    public function stockHistories()
    {
        return $this->hasMany(StockHistory::class);
    }

    public function addStock($quantity, $notes = null)
    {
        $previousStock = $this->stock_quantity;
        $this->stock_quantity += $quantity;
        $this->updateStockStatus();
        
        $this->stockHistories()->create([
            'quantity_added' => $quantity,
            'previous_stock' => $previousStock,
            'new_stock' => $this->stock_quantity,
            'notes' => $notes
        ]);
    }

    public function updateStockStatus()
    {
        if ($this->stock_quantity <= 0) {
            $this->stock_status = 'out_of_stock';
        } elseif ($this->stock_quantity <= $this->low_stock_threshold) {
            $this->stock_status = 'low_stock';
        } else {
            $this->stock_status = 'in_stock';
        }
        $this->save();
    }
}
