<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Role;
use App\Models\Permission;
use App\Models\User;
use App\Models\Brand;
use App\Models\Category;
use App\Models\Subcategory;
use App\Models\Unit;
use App\Models\Product;
use Illuminate\Support\Facades\Hash;

class DatabaseSeeder extends Seeder
{
    public function run(): void
    {
        // Create Roles
        $clientRole = Role::create(['name' => 'Client']);
        $masterRole = Role::create(['name' => 'Master']);
        $adminRole = Role::create(['name' => 'Admin']);

        // Create Permissions
        $permissions = [
            'manage_brands', 'manage_categories', 'manage_subcategories', 
            'manage_units', 'manage_products', 'manage_orders', 
            'manage_reports', 'manage_roles', 'manage_users'
        ];

        foreach ($permissions as $permission) {
            Permission::create(['name' => $permission, 'display_name' => ucwords(str_replace('_', ' ', $permission))]);
        }

        // Assign Permissions to Roles
        $clientRole->permissions()->attach([6, 7]); // manage_orders, manage_reports
        $masterRole->permissions()->attach(range(1, 9)); // all permissions
        $adminRole->permissions()->attach(range(1, 9)); // all permissions

        // Create Users
        User::create([
            'name' => 'Admin',
            'email' => 'admin@admin.com',
            'password' => Hash::make('password'),
            'role_id' => 3
        ]);

        User::create([
            'name' => 'Master Admin',
            'email' => 'master@chefschoice.com',
            'password' => Hash::make('password'),
            'role_id' => 2
        ]);

        User::create([
            'name' => 'John Customer',
            'email' => 'customer@example.com',
            'password' => Hash::make('password'),
            'role_id' => 1
        ]);

        // Create Units
        $units = [
            ['name' => 'kg', 'symbol' => 'kg'],
            ['name' => 'gram', 'symbol' => 'g'],
            ['name' => 'litre', 'symbol' => 'L'],
            ['name' => 'piece', 'symbol' => 'pcs'],
            ['name' => 'dozen', 'symbol' => 'dz']
        ];

        foreach ($units as $unit) {
            Unit::create($unit);
        }

        // Create Brands
        $brands = [
            ['name' => 'Fresh Valley', 'description' => 'Premium fresh produce', 'slug' => 'fresh-valley'],
            ['name' => 'Ocean Catch', 'description' => 'Fresh seafood and fish', 'slug' => 'ocean-catch'],
            ['name' => 'Farm Fresh', 'description' => 'Organic dairy and meat', 'slug' => 'farm-fresh'],
            ['name' => 'Spice Garden', 'description' => 'Authentic spices and herbs', 'slug' => 'spice-garden'],
            ['name' => 'Golden Harvest', 'description' => 'Quality grains and cereals', 'slug' => 'golden-harvest']
        ];

        foreach ($brands as $brand) {
            Brand::create($brand);
        }

        // Create Categories with Subcategories
        $categoriesData = [
            'Vegetables' => ['Leafy Greens', 'Root Vegetables', 'Seasonal Vegetables'],
            'Fruits' => ['Tropical Fruits', 'Citrus Fruits', 'Berries'],
            'Meat & Poultry' => ['Chicken', 'Beef', 'Mutton'],
            'Fish & Seafood' => ['Fresh Fish', 'Frozen Fish', 'Prawns & Crab'],
            'Dairy Products' => ['Milk', 'Cheese', 'Yogurt'],
            'Spices & Herbs' => ['Whole Spices', 'Ground Spices', 'Fresh Herbs'],
            'Rice & Grains' => ['Basmati Rice', 'Local Rice', 'Lentils']
        ];

        foreach ($categoriesData as $categoryName => $subcategories) {
            $category = Category::create([
                'name' => $categoryName,
                'slug' => strtolower(str_replace([' ', '&'], ['-', 'and'], $categoryName)),
                'brand_id' => rand(1, 5)
            ]);

            foreach ($subcategories as $subcategoryName) {
                Subcategory::create([
                    'name' => $subcategoryName,
                    'slug' => strtolower(str_replace([' ', '&'], ['-', 'and'], $subcategoryName)),
                    'category_id' => $category->id
                ]);
            }
        }

        // Create Products
        $products = [
            ['name' => 'Fresh Spinach', 'price' => 2.50, 'category_id' => 1, 'subcategory_id' => 1, 'unit_id' => 1, 'stock_quantity' => 50],
            ['name' => 'Organic Carrots', 'price' => 3.00, 'category_id' => 1, 'subcategory_id' => 2, 'unit_id' => 1, 'stock_quantity' => 40],
            ['name' => 'Bell Peppers', 'price' => 4.50, 'category_id' => 1, 'subcategory_id' => 3, 'unit_id' => 1, 'stock_quantity' => 30],
            ['name' => 'Fresh Mangoes', 'price' => 5.00, 'category_id' => 2, 'subcategory_id' => 4, 'unit_id' => 1, 'stock_quantity' => 25],
            ['name' => 'Sweet Oranges', 'price' => 3.50, 'category_id' => 2, 'subcategory_id' => 5, 'unit_id' => 1, 'stock_quantity' => 35],
            ['name' => 'Fresh Strawberries', 'price' => 8.00, 'category_id' => 2, 'subcategory_id' => 6, 'unit_id' => 2, 'stock_quantity' => 20],
            ['name' => 'Chicken Breast', 'price' => 12.00, 'category_id' => 3, 'subcategory_id' => 7, 'unit_id' => 1, 'stock_quantity' => 15],
            ['name' => 'Ground Beef', 'price' => 15.00, 'category_id' => 3, 'subcategory_id' => 8, 'unit_id' => 1, 'stock_quantity' => 12],
            ['name' => 'Mutton Chops', 'price' => 18.00, 'category_id' => 3, 'subcategory_id' => 9, 'unit_id' => 1, 'stock_quantity' => 10],
            ['name' => 'Fresh Salmon', 'price' => 20.00, 'category_id' => 4, 'subcategory_id' => 10, 'unit_id' => 1, 'stock_quantity' => 8],
            ['name' => 'Tiger Prawns', 'price' => 25.00, 'category_id' => 4, 'subcategory_id' => 12, 'unit_id' => 1, 'stock_quantity' => 6],
            ['name' => 'Fresh Milk', 'price' => 4.00, 'category_id' => 5, 'subcategory_id' => 13, 'unit_id' => 3, 'stock_quantity' => 50],
            ['name' => 'Cheddar Cheese', 'price' => 8.50, 'category_id' => 5, 'subcategory_id' => 14, 'unit_id' => 2, 'stock_quantity' => 20],
            ['name' => 'Cumin Seeds', 'price' => 6.00, 'category_id' => 6, 'subcategory_id' => 16, 'unit_id' => 2, 'stock_quantity' => 30],
            ['name' => 'Turmeric Powder', 'price' => 4.50, 'category_id' => 6, 'subcategory_id' => 17, 'unit_id' => 2, 'stock_quantity' => 25],
            ['name' => 'Basmati Rice Premium', 'price' => 12.00, 'category_id' => 7, 'subcategory_id' => 19, 'unit_id' => 1, 'stock_quantity' => 40],
            ['name' => 'Red Lentils', 'price' => 5.50, 'category_id' => 7, 'subcategory_id' => 21, 'unit_id' => 1, 'stock_quantity' => 35]
        ];

        foreach ($products as $productData) {
            Product::create([
                'name' => $productData['name'],
                'slug' => strtolower(str_replace([' ', '&'], ['-', 'and'], $productData['name'])),
                'description' => "Fresh and high-quality {$productData['name']}",
                'price' => $productData['price'],
                'discount_price' => $productData['price'] * 0.9,
                'sku' => 'SKU' . str_pad(rand(1000, 9999), 4, '0', STR_PAD_LEFT),
                'category_id' => $productData['category_id'],
                'subcategory_id' => $productData['subcategory_id'],
                'brand_id' => rand(1, 5),
                'unit_id' => $productData['unit_id'],
                'stock_quantity' => $productData['stock_quantity'],
                'cost_price' => $productData['price'] * 0.7,
                'status' => 'active',
                'visibility' => 'public',
                'weight' => rand(100, 2000),
                'dimensions' => '10x10x5 cm',
                'tags' => 'fresh,organic,premium'
            ]);
        }
    }
}